////////////////////////////////////////////////////////////////////////////////////////////////////////////////
// MultiRender.cpp: implementation of the CMultiRender class.
//
////////////////////////////////////////////////////////////////////////////////////////////////////////////////

#include "Rays.h"						// Application. LOGGING FLAG..
#include "RaysDoc.h"					// Document.
#include "Vector.h"						// Vectors.
#include "SceneObject.h"				// Scene object.
#include "LightSource.h"				// Source of illumination.
#include "Sphere.h"						// Sphere object.
#include "Plane.h"						// flat object.
#include "Surface.h"					// Surfaces.
#include "MultiRender.h"				// This... rendering enigine,
#include <math.h>						// Standard C mathematics.
#include <assert.h>						// Assert..
#include "RaysView.h"					// The view attached to the window.

// Max recurse level should be user configurable.
#define MAX_RECURSE_LEVEL	15
#define BYTES_PER_PIXEL 4



//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////


CMultiRender::CMultiRender()
{
	SceneList = new BList();
	LightList = new BList();
}

inline void SafeDelete( void *Obj )
{
	if( Obj )
		delete Obj;

	// On the PowerPC uninitialised or deleted pointers get 0xdeadbeef, Windows x86 gets 0xcdcdcdcd
	// just to make sure.
	Obj = NULL;
}

CMultiRender::~CMultiRender()
{
	EmptyScene();	
		
	// clean up the list objects themselves..		
	SafeDelete( LightList );
	SafeDelete( SceneList );
}

//////////////////////////////////////////////////////////////////////
// Clear the scene.

void CMultiRender::EmptyScene()
{
	int NumLights = LightList->CountItems();
	int NumObjects = SceneList->CountItems();

	// Delete all items in the light list.
	for( int CurrLight = 0; CurrLight < NumLights; CurrLight++ )
		SafeDelete( LightList->ItemAt( CurrLight ) );
		
	// Delete all items in the scene list.	
	for( int CurrObject = 0; CurrObject < NumObjects; CurrObject++ )
		SafeDelete( SceneList->ItemAt( CurrObject ) );

	// Clear lists.
	LightList->MakeEmpty();
	SceneList->MakeEmpty();
}



//////////////////////////////////////////////////////////////////////
// Called from the parser, makes a sphere object and inserts it into
// the scene list.
#ifdef __INTEL__
extern "C" void MakeSphere( float x, float y, float z, float Radius, float R, float G, float B,
				 float Ambient, float Diffuse, float Brilliance, float Specular, float Roughness, float Reflection  
			   )
#else
void MakeSphere( float x, float y, float z, float Radius, float R, float G, float B,
				 float Ambient, float Diffuse, float Brilliance, float Specular, float Roughness, float Reflection  
			   )
#endif
{

	CSphere *NewPtr = new CSphere( x, y, z, Radius );
	( (RaysWnd *) be_app->WindowAt(0) )->RenderContext->SceneList->AddItem( NewPtr );
	NewPtr->SetColour( CColour( R, G, B ) );
	NewPtr->SetProperties( Ambient, Diffuse, Brilliance, Specular, Roughness, Reflection );
}

//////////////////////////////////////////////////////////////////////
// A default world..

void CMultiRender::MakeDefault( unsigned int Width,  unsigned int Height )
{
	// default to sky blue.
	Background = CColour( (unsigned char) 175, (unsigned char) 200, (unsigned char) 255 );


	//////////////////////////////////////////////////////////////////////////////
	// Default..


	// Red
	float SphereX = 50.0f;
	float SphereY = 0.0f;
	float SphereZ = 0.0f;
	float SphereRad = 25.0f;
	CSceneObject *TestObj = new CSphere( SphereX, SphereY, SphereZ, SphereRad );
	TestObj->SetColour( CColour( (unsigned char) 220,  (unsigned char) 100,  (unsigned char) 100 ) );
	TestObj->SetProperties( 0.3, 0.7, 2.0, 0.9, 50.0, 0.065 );


	// Clear / Blue
	SphereX = -37.5f;
	SphereY = -15.0f;
	SphereZ = -15.5f;
	SphereRad = 15.0f;
	CSceneObject *TestObj2 = new CSphere( SphereX, SphereY, SphereZ, SphereRad );
	TestObj2->SetColour( CColour( (unsigned char) 120, (unsigned char) 200, (unsigned char) 220 ) );
	TestObj2->SetProperties( 0.3, 0.7, 20.0, 0.9, 100.0, 0.75 );

	// Yellow
	SphereX = 5.5f;
	SphereY = -42.5f;
	SphereZ = 5.0f;
	SphereRad = 17.5f;
	CSceneObject *TestObj3 = new CSphere( SphereX, SphereY, SphereZ, SphereRad );
	TestObj3->SetColour( CColour( (unsigned char) 229, (unsigned char) 220, (unsigned char) 0 ) );
	TestObj3->SetProperties( 0.9, 0.8, 20.0, 0.9, 0.85, 0.0 );

	// DEBUG planes..
	CSceneObject *Floor = new Plane( 0, 1, 0, -50 );
	Floor->SetColour( CColour( (unsigned char) 150, (unsigned char) 175, (unsigned char) 150 ) );
	Floor->SetProperties( 0.3, 0.7, 20.0, 0.9, 100.0, 0.65 );
	CLightSource *Light;

	Light = new CLightSource( CVector( 65, -50, -100) );
	Light->SetColour( CColour( 1.0f, 1.0f, 1.0f ) );



	// DEBUG code..
	SceneList->AddItem( TestObj );
	SceneList->AddItem( TestObj2 );
	SceneList->AddItem( TestObj3 );
	SceneList->AddItem( Floor );

	LightList->AddItem( Light );

}


//////////////////////////////////////////////////////////////////////
// For each row of the image
//		for each column of the image.
//			Generate ray for view plane thru row, column..
//			for each object in the document.
//				Calculate ray/object intersection.
//				If the intersection occured and intersect time is smallest so far
//					Save intersection time.
//
//		if intersect
//			Shade...
//
//
//



void CMultiRender::Go( CRaysDoc *pDoc, unsigned int Width,  unsigned int Height, RaysWnd *MainWnd )
{
	time_t StartTime = time( NULL );				// Start timer..
	unsigned long RendTime = 0;						


	/////////////////////////////////////////////////////////////////////
	//	Set rendering dimentions by probing the window client area.

	nWidth = Width;
	nHeight = Height;
	float AspectRatio = float( nWidth )/ float(nHeight);

	((CRaysView *)MainWnd->FindView( "Rays" ))->Clear();

	bool WasEmpty; 
	WasEmpty = false;

	// Empty scene so create a default one for debug or demo.	
	if( SceneList->IsEmpty() )
	{
		MakeDefault( Width, Height );
		WasEmpty = true;
	}

	// Create a camera or eye point.
	CVector Direction( 0.0f, -0.05f, 1.0f );
	CVector Location( 0.0f, -5.0f, -110.0f );
	CVector	Right( 1.0f, 0.0f, 0.0f );
	CVector	Up( 0.0f, AspectRatio, 0.0f );
	ViewPoint Camera( Direction, Location, Right, Up );


	//////////////////////////////////////////////////////////////////////////////
	//	Heart of the ray tracer..

	// Create image..
	pDoc->OutputImage.SetSize( nWidth, nHeight );

	for( unsigned int row = 0; row < nHeight; row++ )
	{
		for( unsigned int col = 0; col < nWidth; col++ )
		{
			RecurseLevel = 0;									// Number of calls to Trace().
			Ray TracedRay;										// The Ray to trace.
			CColour Pixel;										/// The colour to set.
			MakeRay( row, col, TracedRay, Camera );				// Create Starting view point.
			Trace( TracedRay, Pixel );							// Do the actual tracing of the ray.
			SetColour( col, row, pDoc->OutputImage, Pixel, MainWnd );	// Set the result.
		}
		((CRaysView *)MainWnd->FindView( "Rays" ))->DrawBitmap( ((CRaysView *)MainWnd->FindView( "Rays" ))->OutImg );
	}

	// Started with a default scene, loads should not merge into it. leave it clean.
	if( WasEmpty )
		EmptyScene();	

	//////////////////////////////////////////////////////////////////////
	// Clean up..
	//delete PrimaryLight;


	//////////////////////////////////////////////////////////////////////
	// Finished message,
	RendTime = time(NULL) - StartTime;
}
 





////////////////////////////////////////////////////////////////////////
//

void CMultiRender::MakeRay(int X, int Y, Ray& TheRay, ViewPoint EyePos )
{
	CVector Temp1, Temp2;
	double ScaledX, ScaledY;	// Screen co ordinates to converted to world.

	// Should Range between -0.5 to 0.5..
	ScaledX = ( (double) X - (double) nWidth / 2.0 ) / (double) nWidth;
	ScaledY = ( (double) Y - (double) nHeight / 2.0 ) / (double) nHeight;

	// Debug code...
	assert( (ScaledX <= 5.0 ) && ( ScaledY <= 5.0 ) );					// Bound from above.
	assert( (ScaledX >= -5.0 ) && ( ScaledY >= -5.0 ) );				// Bound from below.

	// Work out direction...
	Temp1 = EyePos.Up *  ScaledX;
	Temp2 = EyePos.Right * ScaledY;
	TheRay.Direction = Temp1 + Temp2;
	TheRay.Direction = EyePos.Direction + TheRay.Direction;
	~TheRay.Direction;													// Normalise..
	TheRay.Origin = EyePos.Location;									// Origin at View point.

}


//////////////////////////////////////////////////////////////////////////////
//
//	Trace member function: traces the ray.
//
// returns: TRUE if hit an object. FALSE otherwise.


bool CMultiRender::Trace(Ray& TheRay, CColour& RgbVal )
{
	RgbVal = Background;											// Default: Background colour ..

	if( RecurseLevel > MAX_RECURSE_LEVEL )
		return FALSE;

	CSceneObject *ClosestObject = NULL;
	auto bool Intersected = FALSE;

																	// For every object, check for intersection.
																	//	Save the closest objtect..
	CSceneObject *CurrObject = NULL;
	if( SceneList->IsEmpty() )										// Empty scene..
	{
		return FALSE;
	}
		
	CurrObject = (CSceneObject *) SceneList->FirstItem();
	double ClosestTime = 999999.99;

	int	pos;
	
	pos = 0;
	bool EndOfList = FALSE;
	while( ! EndOfList )
	{
		double InVal;

		if( !CurrObject )											// Is this the last object ?
			EndOfList = TRUE;

		if( CurrObject )											// Valid object ?	
		{
			InVal = CurrObject->Intersect( TheRay );				// Intersects ?

			if( (InVal > EPSILON ) && (InVal < ClosestTime) )		// Closer ?
			{
				ClosestObject = CurrObject;							// Save..
				ClosestTime = InVal;
				Intersected = TRUE;
			}
		}
		CurrObject = (CSceneObject *) SceneList->ItemAt( pos++ );
	}
					
	if( Intersected )												// We hit something....
	{
		CVector Normal;
		double NormalDir;


		CVector IntersectPoint( ClosestTime * TheRay.Direction.GetX() + TheRay.Origin.GetX(),
								ClosestTime * TheRay.Direction.GetY() + TheRay.Origin.GetY(),
								ClosestTime * TheRay.Direction.GetZ() + TheRay.Origin.GetZ() 
							  );	

		Normal = ClosestObject->Normal( IntersectPoint ); 
		NormalDir = Normal %  TheRay.Direction;					// Dot product of Normal and Ray.
		if( NormalDir > 0.0f )									
			-Normal;

		Shade( ClosestObject, TheRay, Normal, IntersectPoint, RgbVal );
	}
		
	return Intersected;
}


//////////////////////////////////////////////////////////////////////////////
// Set colour..

void CMultiRender::SetColour(int X, int Y, CImage& Image, CColour Value, RaysWnd *MainWnd )
{
	// Offset into the image.
	int Offset = (Y * (Image.GetWidth() + 1) * BYTES_PER_PIXEL ) + abs((X -1) * BYTES_PER_PIXEL);
	
	// Pack the pixel.
#ifndef __INTEL__
	int32 RGBValue = (Value.GetR() << 24 ) | (Value.GetG() << 16) | (Value.GetB() << 8) | 0xff;
#else
	int32 RGBValue = (Value.GetB() << 16 ) | (Value.GetG() << 8) | (Value.GetR());
#endif
	// Set value.
	((CRaysView *)MainWnd->FindView( "Rays" ))->OutImg->SetBits( (const void *) &RGBValue, (int32)1, (int32) Offset, B_RGB_32_BIT );
	
	// For later processing on a by channel bais..
	Image.SetPixel( X, Y, Value );	
}


//////////////////////////////////////////////////////////////////////////////
// Shade. 
//


void CMultiRender::Shade(CSceneObject * ClosestObj, Ray TheRay, CVector Normal, CVector Point, CColour& col)
{

	// If no intersection just return. A small optimisation.
	if( !ClosestObj )
		return;

	double K, Ambient, Diffuse, Specular, DistanceT;
	Ray LightRay;
	Ray ReflectedRay;
	double Red, Green, Blue;	// Working RGB values.. need to be onverted to a CColour later.	

	int32 NumLights;			// 

	CLightSource *Light;
	K = TheRay.Direction % Normal;					/// Dot product of the ray and the objects normal.
	K *= 2.0;

	// Reflect ray (obviousily) starts at the intersection point.
	ReflectedRay.Origin = Point;

	ReflectedRay.Direction = CVector(  K * Normal.GetX() + TheRay.Direction.GetX(),
									   K * Normal.GetY() + TheRay.Direction.GetY(), 
									   K * Normal.GetZ() + TheRay.Direction.GetZ()
									);
	Ambient = ClosestObj->Properties.Ambient;
	Red = ClosestObj->Properties.Colour.Rval * Ambient;
	Green = ClosestObj->Properties.Colour.Gval * Ambient;
	Blue = ClosestObj->Properties.Colour.Bval * Ambient;

	// Iterate thru a Light list. 
	// DEBUG for now use a bogus one.

	Light = new CLightSource( CVector( 65, -50, -100) );
	Light->SetColour( CColour( 1.0f, 1.0f, 1.0f ) );

	NumLights = LightList->CountItems();
	
	if( Light )
	{
		CColour LightColour;
		DistanceT = Light->MakeLightRay( Point, LightRay );
	
		if( RecurseLevel == 0 )
			LightColour = Light->GetColour(ClosestObj, LightRay, DistanceT, SceneList );
		else
			LightColour = Light->Properties.Colour;

		if(  (LightColour.Rval == 0.0) || (LightColour.Gval == 0.0) || ( LightColour.Bval == 0.0) )
		{
//			col = CColour(  0.0f, 0.0f, 0.0f );
//			return;
		}
		else
		{
			Diffuse = Normal % LightRay.Direction;												// Dot Product 

			if( ( Diffuse > 0.0 ) && ( ClosestObj->Properties.Diffuse > 0.0 ) )
			{
				Diffuse = pow( Diffuse, ClosestObj->Properties.Brilliance ) * ClosestObj->Properties.Diffuse;
				Red += LightColour.Rval * ClosestObj->Properties.Colour.Rval * Diffuse;
				Green += LightColour.Gval * ClosestObj->Properties.Colour.Gval * Diffuse;
				Blue += LightColour.Bval * ClosestObj->Properties.Colour.Bval * Diffuse;
			}

			~ReflectedRay.Direction;	//
			Specular = ReflectedRay.Direction % LightRay.Direction;							// Dot product..


			if( (Specular > 0.0 ) && ( ClosestObj->Properties.Specular > 0.0 ) )
			{
				Specular = pow( Specular, ClosestObj->Properties.Roughness ) * ClosestObj->Properties.Roughness;
				Red += LightColour.Rval * Specular;
				Green += LightColour.Gval * Specular;
				Blue += LightColour.Bval * Specular;
			}

			// Reflection ..
			K = ClosestObj->Properties.Reflection;
			if( K > 0.0 )
			{
				CColour NewColour;
				RecurseLevel++;

				Trace( ReflectedRay, NewColour );		// Recursive..
				Red += NewColour.Rval * K;
				Green += NewColour.Gval * K;
				Blue += NewColour.Bval * K;

				RecurseLevel--;
			}

			/// Lights out !
			delete Light;
		}
	}
	col = CColour( (float) Red,  (float) Green, (float) Blue );
}




